# Copyright 2008, Valve Corporation
#
# Name :          ht.rb "Hammer Time" 1.0
# Description :   Exports SketchUp models to Hammer, the Source Engine Editor (Halfe Life 2, Team Fortress 2, Portal)
# Author :        Yasser Malaika
#
# Install:        1. Place this file, 'ht.rb' in this folder: C:\Program Files\Google\Google SketchUp 6\Plugins
#                 2. Place the file '1REFTEX.jpg' in the same folder
#
# Usage :         From SketchUp menu, select Plugins/Hammer Time!. Groups and Components are exported as Brushes and must be convex,
#                 components with recognized names are exported as entities (light, light_env, player_start, etc.)
# Date :          27.May.2006 - July 2008
# Type :          Exporter
# History:        0.0 (27.May.2006) - Start development on proof of concept
#                 0.1 (3.July.2006) - groups, components, and basic UV working
#                 0.2 (10.July.2006) - Sent POC to Chris Chin for testing, feedback
#                 0.3 (1.September.2006) Began revising POC to be used as a real plugin.
#                 0.4 (19.February.2008) The project is happening! I should dust this off, update the comments,
#                                        and see if it can help me get back into source content creation.
#                 0.5 (28.February.2008) Nesting Groups now work. material and transform pass-through. Next step: implement UVW hack I did before
#                                        new parsing function is recursive, thus supports groups within groups,
#                                        and also passes down transforms, materials to accumulate, etc.
#                 0.6 (05.March.2008) Basic UV is working. Am still trying to tease out the nuances of Hammer's UV system.
#                                     It looks as though (racking or skewing) the UV vectors is a no-no.
#                 0.7 (02.June.2008) Chris says NDA issues are holding up official plug-in. So I'll open this
#                                    back up to see if we can make orange level creation move a little faster
#                 0.75(19.June.2008) Added a world size check. Added code to tell you the offiending dimension. This will be usefil for auto skybox, as well
#                 0.76(20.June.2008) Added support for component instances. Now supports arbitrary group/component hierarchies. Now I need to support hammer group IDs
#                 0.77(23.June.2008) Changing entities and prefabs as a second pass, as Hammer does not include them in the world, but I need to keep transforms and group ID relationships
#                 0.78(21.July.2008) Verbose console output is causing dramatic slowdown on larger models and is disabled by default
#                                    Making parser more flexible to handle multiple passes. Now does analysis pass so that we can estimate time, autocomplete, etc.
#                                    Added a time estimate to the status bar. Taking a pass at writeface() perf.
#                 0.79(24.July.2008) Rewriting geometry algorithms has already resulted in dramatic perf gains. Models that were taking hours now process in seconds.
#                                    Unfortunately, mirroring, textures, and who know what else is broken again.
#                 0.80(28.July.2008) Added support for mirroring with hierarchical nesting
#                 0.81(30.July.2008) Decoupled Group ID's from SketchUp Group entity ID's - SU ID's were too bloated for hammer, and were causing incorrect groupings for large
#                                    numbers of brushes and/or complex nestings.
#                                    Also, added popup alerts for problem geometry: singlly grouped groups, brushes with less than 4 sides, lines, etc.
#                                    Fixed the save dialog not prompting for file overwrite
#                                    modified warning to output usefully to console, separated debug, warn, and error popups.
#                                    Bug fix: single object groups no longer currupting Group ID (In theory)
#                                    Added option to use slow algorithm, since slowdown is caused by outliner and component inspectors, and old alg is still more robust due to SU engine oddness with mirror transforms
#                 0.82(12.November.2008) Added support for proxies: can now set 2D skybox by component
#                                    Added code in preparation for auto 3d skybox.
#                                    Fixed reftex needing to be in fixed location. can now live in plugins/hammertime folder with icon and other resources. (finally)
#                                    Updated new fast fast writing with basic UV, added option to support 3 UV levels, none, fast, accurate



#Notes:
#basic usability:
#UV export in fast mode

#basic functionality: save out a model to be loaded into hammer, or as a level, with auto box, player start, etc.
#Export
#  -Geometry
#  -Map (Left 4 Dead)
#  -Map (Half Life 2)
#  -Map (Team Fortress 2)

#Need to have for plug and play usability mode:
#-Auto extents solids
#-auto player start
#-auto light
#-uvmapping

#Awesome wintonium
#-export prefabs
#-export textures as vtf


#Notes:

#Limitations:
# L1. very small, or micro brushes, do not work well. They seem to be fine in hammer, but turn into huge random planes when compiled
# Possible fix: filter out objects with small bounding boxes and output them as a static mesh model instead of a CSG brush. check
# along both local and global coord system, just to catch most issues.

# L2. enclosing Skybox has to be fully sealed, is easy to break, and any volumes that protrude can cause problems.
# Possible fix: automatically generate a skybox by default. If user adds a sky box object, use that instead. should use nodraw on outside

#L3a. (FIXED!) Only top level groups and components will work. this makes it difficult to manage a model of any useful complexity.
#    fix: added a recursive parser function that supports transforms,layers and material pass-down. This provides WYSIWYG output for hammer groups, materials, etc.

#L3b How to do the ID's?
#ID options:
# Group with geometry only - output as a solid
# Group with other groups in it only - output as a group ID
# Identified named component- use prefix to do ent, prefab, or other proxy
# Component with geometry only - brush
# Component with objects only - a group ID
# Brush object that is too small to be a brush - model instead
# group or component with Force model, skybox, or nodraw layer - model, 3D skybox, and ignored.

#L4. auto material generation and compilations-Save texture as TGA a nd save out the VMT
#Ruby should be able to run the natural compiler as well.

#L5 The world size is limited.
# possible fix: create a 3D sky box automatically. Tag either layer or component name "3Dskybox".
# These are left out of extents calc, then processed separately at 1/16th scale outside the level extents
# 2D skybox is also possible

#L6 (FIXED!)Should be able to place arbitrary entities, rather than requiring that they be explicitly supported in the script
#   fix: Added a second pass that adds entities by name. Uses Prefix of 'ENT.' in SU components...

#L7 (FIXED!)It's slow... Some SU classes don't work as described in the SU docs. I think I can find faster workarounds by eliminating the entitiy creation steps.
# Workaround: Close those darned inspectors. Outliner is a major culprit, as is component browser.
# fix: calculate UV plane intersections and geometry transforms directly, instead of using SU drawing functions


#Bugs:
#B1. non-uniform as well as  "skew" or complex transforms, will break the lighting pass, making large black splotches appear or even making VRAD crash.
#    I suspect that VRAD doesn't like non-orthogonal UVs...
#    Workaround:  explode, then regroup the brush.
#    Possible fix: change the UV algorith to world projection automatically when skew transforms are present.
#    Possible fix: make the plug-in explode and regroup for you as neccesary when skew transforms are detected.

#B2. Mixing entities and brushes doesn't properly group
#    Workaround: don't mix them
#    Possible fix: Track what hammer ID's are assigned to each grouping


#---------------------------------------------------------------------------------


#For release, these modes will be used to specify output to a certain game, or just a plain model via the menu.

def ht #Hammer Time!
    $brush_mode="fast"
    mainfunc()
    end

def htslow #Hammer Time!
    $brush_mode="slow"
    mainfunc()
    end




def mainfunc #Hammer Time!

#Output debug information to a pesky alert dialog
$debugAlert=false

#Output warning information to a pesky alert dialog
$warnAlert=true

#Output error information to a pesky alert dialog
$errorAlert=true

#Output info to the Ruby Console. Use only to debug, as this will slow down the calc and cause UI Freezing on large files)
#$consoleOutput=false
$consoleOutput=true

#Output debug information to the output file itself. (resulting file is not readable by hammer.)
# Need to modify this so that the info is written out as a comment
$debugOutput=true

#prompts with a file dialog if true, puts "testmap.vmf" to yasser's desktop if false
usefiledialog=false

#Calculate UVW axis aligned to each face.
$uv_mode=1
# 0 = Ignore UVs, don't output. Won't compile, but assuming that user wants to UV themselves.
# 1 = Use face normal-Simple, fast, but doesn't look like SU model
# 2 = Maintains xform, scale, and UV displacement

#The master default material. Eventually, can automatically assign different colors for floors, walls.
defaultmaterialname= "dev/graygrid"
#defaultmaterialname= "MODELS/WORLDCRAFT/NODES/GROUND_NODE_HINT"

defaultmaterial_width= 512
defaultmaterial_height= 512

#The default skybox.
$defaultskyboxname= "sky_day01_05"           #Half Life 2 default
#$defaultskyboxname= "????????????"           #Team Fortress 2 default
#$defaultskyboxname= "urbannightstormhdr"      #Left 4 Dead


$global_texture_scale_factor=1.125
# Seems to be a standard default in hammer.
# (update) from source wiki: At the default Texturemap scale (0.25) 1 texture pixel = 0.25 x 0.25 units (or 4x4 texels = 1 grid unit).
# Eg: an (undistorted) 512 x 512 texture covers 128 x 128 game units.
# At the default Lightmap scale (16) 1 lightmap pixel = 16 x 16 units.
#
# So the goal is to use the above with material coordinates+uv projection transform to generate
# coordinates that correctly represent those in SU
# useful for auto-generating TGA



#==========
#Initilaize
#==========

# to support various versions with one plugin
$su_version = Sketchup.version_number.to_f / 1000000

#String that describes the plugins folder
$pluginsfolder = Sketchup.find_support_file("Plugins")

model = Sketchup.active_model
$tempmodel =  Sketchup.active_model

# Signal start of Undo sequence event
model.start_operation "Export Model"



#==========
#Open Files
#==========

# Prompt for Output file...
if usefiledialog
    out_name = UI.savepanel( "Export To Source Hammer vmf", "." , "export.vmf" ) #use the dialog (for released version)
else
    out_name = "C:\\Documents and Settings\\ymalaika\\Desktop\\testmap.vmf" #just spit it out here (for development)
    #out_name = "C:\\Documents and Settings\\yasser\\Desktop\\testmap.vmf" #just spit it out here (for development)
end

$export_file = File.new( out_name , "w" )

#if $debugAlert
#   alertmessage= "saving to file: "+out_name
#   UI.messagebox alertmessage
#end

#Write to SketchUp Status Bar
statusBarPreText="   Exporting to Hammer VMF format - Analyzing..."
Sketchup.set_status_text( statusBarPreText+"..." )

# output to Ruby Console
puts( "___________________________________________________________________________" )
puts( "Exporting to Hammer VMF..." )
puts( "   Output File: "+out_name )


$nextavailable_id = 2
# Keep track of Hammer Entity ID's
# We could just use SU internal ID's, but since these include every face and edge,
# they don't reflect what's important to Hammer and results in significant ID bloat)

$file_level = 0
# Keep track of the model hierarchy level

#global arrays that keeps track of material stack. Used while parsing the model hierarchy to provide default material override
$currentmaterial=[]
$currentmaterial_name=[]
$currentmaterial_width=[]
$currentmaterial_height=[]

#If no materials at all are specified, default to the master defualt
$currentmaterial[0]
$currentmaterial_name[0]=defaultmaterialname
$currentmaterial_width[0]=defaultmaterial_width
$currentmaterial_height[0]=defaultmaterial_height



#A global array that keeps track of transfom stack. Used to put evertything where it belongs while parsing the model hierarchy, to rebuild the current transform when needed.
$currentxform=[]

#A global array that keeps track of the mirror xforms through the hierarchy. Use to change winding order of polygons
$currentxformismirrored=[]
$currentxformismirrored[0]=false
$localmirror=false

#A global array that keeps track of Group/component parenting. This maintains group hierarchy in hammer.
$currentGroupID=[]
$grouptoHammerIDmapping=[]




#=========================================================
#Parse objects for analysis
#=========================================================

#Gather info about model: total bounds, bounds of non 3d skybox info, number of solids, ents, prefabs, skybox texture,
$ParsingMode = "analyze"

#Add top level entities to total for indication on the status bar
entCnt=model.entities.count

#puts( "Total Entities: "+entCnt.to_s ) #This only gives the top level entities. We need the full count to do meaningful progress
puts( "" )
puts( "Analyzing Model..." )

#Make sure the model is not empty
#ToDo: just export a simple compileable map
if modelisempty()
    puts( "\nERROR: Export Failed: Model appears empty." )
    if $errorAlert then UI.messagebox "Error: The SketchUp model appears to be empty.\n\nMake a box, group it, and try again." end
       return false
    end

#Store extents/bounds values. Used for creating autoskybox
boundingbox = Geom::BoundingBox.new

# Get model bounds
boundingbox = boundingbox.add model.bounds

# Largest allowed map without 3D skybox is 32768 units. 16384 max unit depth.
bbmin=Geom::Point3d.new boundingbox.min
bbmax=Geom::Point3d.new boundingbox.max


# Check to make sure within Engine limits

if ValidHammerMaxBounds( boundingbox )
    #Model extents are within hammer limits
    puts( "   min (Neg) extents: "+bbmin.to_s )
    puts( "   max (Pos) extents: "+bbmax.to_s )
    #puts( "Bounds OK" )
    puts( "" )
else
    #Model extents exceed hammer limits
    puts( "Error: Maximum dimensional extents of Hammer exceeded! Export Failed." )
    # ValidHammerMaxBounds spits out a more useful alert
    #if $errorAlert then UI.messagebox "Error: The SketchUp model dimensions are too big for Hammer. Try exporting a smaller model." end
    return false
end
# A smart appraoch would be to analyze non-3d Skybox data, and use that instead. The outlyers would be made into a skybox. This effectivly increases the allowable size of visible data.


$totalEntsInWorld=entCnt     #All entities

$totalUsefulEntsInWorld=0    #All entiteis that can be processed into useful output

$totalBrushesInWorld=0      #solid brush geometry
$totalPlaceableEntitiesInWorld=0  #

$totalTreeGroupsInWorld=0   #containers like groups and component instances
$totalPrefabModelsInWorld=0        #static model prefabs, to be exported as SMD/DMX files.

$totalProxiesInWorld=0     #Use to track things like skybox texture, etc. Things that are not physically converted, but hold data.

$totalUnrecognizedInWorld=0
$totalOrphanedFacesInWorld=0
$totalSinglyGroupedGroupInWorld=0

$atLeastOne_PlayerStart_Found=false
$atLeastOne_Light_Found=false


for ent in model.entities
    parse(ent)
end

puts( "   Total Objects: " + $totalUsefulEntsInWorld.to_s )
puts( "\n   Groupings................ " + $totalTreeGroupsInWorld.to_s )
puts( "   Brushes.................... " + $totalBrushesInWorld.to_s )
puts( "   Entities:.................... " + $totalPlaceableEntitiesInWorld.to_s )
puts( "   Prefab Models.......... " + $totalPrefabModelsInWorld.to_s )
puts( "   Proxies:.................... " + $totalProxiesInWorld.to_s )
puts( "   Ungrouped Faces:... " + $totalOrphanedFacesInWorld.to_s )
puts( "   Unknown:................ " + $totalUnrecognizedInWorld.to_s )
puts( "" )

# To Do: If 1st parsing pass does not find an "EXT" extents proxy component, automatically make encapsulating brushes
# To Do: If 1st parsing pass does not find an "playerstart" component, automatically make one
# To Do: If 1st parsing pass does not find any "light" components, automatically make one

if $atLeastOne_PlayerStart_Found == false
   puts( "   WARNING: no info_player_start entities exported"  )
   end

if $atLeastOne_Light_Found == false
   puts( "   WARNING: No light entities exported"  )
   end


# Trying to close or minimize inspector,
#it considers open=visible, and closing works (it minimizes it)
#it considers min=nonvisible, so it opens it. Very poor.
# thus, the ony way to get around the problem is to close it!!!!!!!  Argh stupid SketchUp API

#if isWindowVisible("Outliner")
#   UI.show_inspector "Outliner"
#end

#=============
#Start Writing
#=============

#write out the header info
output_header


#=========================================================
# Parse for brushes
#=========================================================
  $remainingEnts=$totalUsefulEntsInWorld
  $percentComplete=0.0
  
  $blinker=1

  $time_started=0.0
  $time_started=Time.now.to_f
  #puts( $time_started  )
  $previous_time_estimate=0.0
  $previous_percent_estimate=0.0

  $ParsingMode = "outputbrushes"

  if $totalBrushesInWorld > 0

    puts( "\nWriting Brushes..." )
    puts( "" )

    for ent in model.entities
      UpdatePercentComplete()
      parse(ent)
    end

  else
    if $warnAlert then UI.messagebox "Exporter Warning: There don't appear to be valid solid brushes." end
    puts( "WARNING: No Valid Solid Brushes" )
  end

$export_file.puts( "}" )  # Close out the world

if $debugOutput
   $export_file.puts( "// *** [0] End World" )
   $export_file.puts( "" )
   end

#=========================================================
# Parse for Static Models
#=========================================================

puts( "\nWriting Models..." )
$ParsingMode = "outputmodels"

# Model writing Code Here

#=========================================================
# Parse for entities
#=========================================================

puts( "\nWriting Entities..." )
puts( "" )
$ParsingMode = "outputentities"

$remainingEnts = $totalPlaceableEntitiesInWorld
#puts( "*** Non Static Ents:"+$totalPlaceableEntitiesInWorld.to_s  )

if $totalPlaceableEntitiesInWorld > 0
   for ent in model.entities
      #UpdatePercentComplete()
      parse(ent)

#      # Update progress bar
      $percentComplete=( 1.0-( $remainingEnts.to_f/$totalPlaceableEntitiesInWorld.to_f) )*100
      Sketchup.set_status_text("  Exporting to Hammer VMF format - Entities: "+$percentComplete.to_i.to_s+"%" )
      $remainingEnts=$remainingEnts-1
      end

   end



$export_file.close

$time_ended=Time.now.to_f
$time_totalprocessing=$time_ended-$time_started

elapsed=$time_totalprocessing.to_i.to_s
puts( "\nExport Complete. Elapsed Time: "+elapsed + " Seconds"  )
puts( ""  )



if $totalSinglyGroupedGroupInWorld > 0
   puts( "WARNING: Detected Group/Component which contains just one Group/Component. ("+$totalSinglyGroupedGroupInWorld.to_s+" instances)"  )
   puts( "This may produce invalid brush geometry."  )
   puts( ""  )
   if $warnAlert then UI.messagebox "Hammertime Warning: Group contains just one object" end
end

if $totalUnrecognizedInWorld > 0
   puts( "WARNING: Unsupported Geometry detected. ("+$totalUnrecognizedInWorld.to_s+" instances)"  )
   puts( ""  )
   if $warnAlert then UI.messagebox "Hammertime Warning: Unsupported Stuff!" end
end

if $totalOrphanedFacesInWorld > 0
   puts( "WARNING: Orphaned Faces detected. ("+$totalUnrecognizedInWorld.to_s+" instances)"  )
   puts( ""  )
   if $warnAlert then UI.messagebox "Hammertime Warning: Orphaned Faces. Group those guys!" end
end


#SketchUp Status Bar
Sketchup.set_status_text("Export to Hammer VMF format complete. "+out_name )

# Signal end of Undo sequence event
model.commit_operation

end #ht Hammer Time!













# examine an entity for type and content, extract needed info, recursively if required,
# Will do different things, depending on the mode. I had many different parsing functions, but they were similar enough that it became
# cumbersome to edit them all. Need to make the single hairy monster, see what it is, then split it up more smartly.

def parse( ent )
    entid=ent.entityID

    parsevalid=true

    if( ent.is_a? Sketchup::Group or ent.is_a? Sketchup::ComponentInstance ) #--------------------------------- Found a GROUP/COMP

        # Add to total objects to analyze
        if( $ParsingMode == "analyze" ) then $totalUsefulEntsInWorld+=1 end

        $file_level = $file_level + 1
        indent ( $file_level )
  
        materialOfGroup = get_material_name( ent )   # The material currently specified
        materialName=materialOfGroup  # The material to assign. Initialize to currently specified
  
        # If no material is specified, pass down the closest specified one in the hierarchy.
        if( materialOfGroup == "_unspecified_" )
            $currentmaterial_name[ $file_level ] = $currentmaterial_name[ $file_level-1 ]
            materialName = $currentmaterial_name[ $file_level ]
        # If it is, keep track of it for downstream use as a default
        else
            $currentmaterial_name[ $file_level ] = materialOfGroup
            end

#        #DEBUG: Write out the material proccessing
#        $export_file.puts( "" )
#        indent ( $file_level )
#        $tempstring=$tempstring+"// ["+$file_level.to_s+"] Group/Component "+entid.to_s+", mat: " + materialofgroup
#
#        if( materialofgroup == "_unspecified_" )
#          $tempstring=$tempstring + " (using cm" + $file_level.to_s + ": " + materialname +")"
#        else
#          $tempstring=$tempstring + " (setting cm"+$file_level.to_s+" to "+ materialofgroup +")"
#          end
#
#        $export_file.puts( $tempstring  )


        #Store the transform for the current level
        $currentxform[ $file_level ] = ent.transformation

        transform = Geom::Transformation.new $currentxform[ $file_level ]
        transformarray = transform.to_a

        #Debugging: Write out the transformation matrix to the file
        #$export_file.puts( "Level: "+$file_level.to_s )
        #spitoutXform(transformarray)

        #check to see if the local xform is a mirror
        if xformarray_is_mirror(transformarray)
           $localmirror=true
           $currentxformismirrored[ $file_level ]= true
           #$export_file.puts( "Mirror for {"+$file_level.to_s+"} set to TRUE" )
           #puts( $file_level.to_s+" TRUE" )
        else
           $localmirror=false
           $currentxformismirrored[ $file_level ]= false
           #$export_file.puts( "Mirror for {"+$file_level.to_s+"} set to FALSE" )
           #puts( $file_level.to_s+" False" )
           end

        #Check the xform mirror stack


#       #DEBUG: Write out the transform... Integerizing for debugging clarity
#       if $debugOutput
#          temptrans = (get_component_trans ent)
#          indent ( $file_level )
#          $tempstring=$tempstring + "   Local Translation x: " + temptrans[0].to_i.to_s + " y:" + temptrans[1].to_i.to_s + " z:" + temptrans[2].to_i.to_s
#          $export_file.puts( $tempstring  )
#
#          #Heck, write out the whole xform for good measure...
#
#          $export_file.puts( "" )
#          $export_file.puts( "   Local 4x4 transform:" )
#          $export_file.puts( ent.transformation.to_a )
#          end


      entsToSolid=[]
      entsToParse=[]


      #Screen out named entities saved as components   --------------------------------- Found a Named Entities
      if( ent.is_a? Sketchup::ComponentInstance )
          defin=ent.definition
          compdefname = defin.name

          # Pick out named Components Here
          # check name for Supported Special objects, (entities,)

          # extract prefix of string
          defPrefix = compdefname[0..3]
          defName = compdefname[4..-1]

          if ( defPrefix == "ENT." )
             #Component is an entitiy

             if( $ParsingMode == "analyze" ) 
               $totalPlaceableEntitiesInWorld+=1
               countEntity( ent, defName )

             elsif( $ParsingMode == "outputentities" )
                #$tempstring+="{"+$file_level.to_s+"} "
                #$tempstring+="component "+entid.to_s+", mat: " + materialOfGroup +" '" + compdefname +"'"
                indent ( $file_level )
                $tempstring+="["+$file_level.to_s+"] "
                $tempstring+="[ Component "+entid.to_s+" ]"
                $tempstring+="\""+defName +"\"   -->ENTITY"
                puts( $tempstring )

                writeEntity( ent, defName )

                end

             $file_level = $file_level - 1
             return "entity"

          elsif ( defPrefix == "PRE." )
             #Component is a prefab model
             
             if( $ParsingMode == "analyze" ) 
                 $totalPrefabModelsInWorld+=1
             else
               indent ( $file_level )
               $tempstring+="["+$file_level.to_s+"] "
               $tempstring+="< prefab model: \""+defName +"\" postpone to second pass >"
               puts( $tempstring )
                 end
             
             $file_level = $file_level - 1
             return "prefab"

          elsif ( defPrefix == "SKY." )
             #Component is a proxy for the 2D sky box texture. There should only be one per file. more will clobber.

             if( $ParsingMode == "analyze" ) 
                 $totalProxiesInWorld+=1
                 $defaultskyboxname = defName
                 $tempstring= "    Proxy: 2D skybox set to \"" + defName.to_s + "\""
                 puts( $tempstring )
                 puts( "\n" )
                 end

             $file_level = $file_level - 1
             return "2Dsky"

          elsif ( defPrefix == "EXT." )
             #Component is a proxy for 3D extents. There should only be one per file. more will clobber.

             if( $ParsingMode == "analyze" )
                 $totalProxiesInWorld+=1
                 $tempstring= "    Proxy: 3D extents"
                 puts( $tempstring )
                 puts( "\n" )
                 end

             $file_level = $file_level - 1
             return "3dextents"

          else

            #Component is not recognized, so I'm assuming that it contains other components, groups, faces, or combo
            entsToParse = defin.entities

          end


      elsif( ent.is_a? Sketchup::Group )
          entsToParse = ent.entities

      else
             #Should never get to here
             return "prefab"
             end


      #Analyze contents to better handle various conditions.
      # Groups with faces= collect faces for output as solid brush. Groups with Groups - recursively parse Named Groups - that entity

      faces_were_found=false
      faces_in_group=[]
      groups_were_found=false
      groups_in_group=[]
      nonSupportedEntities_in_goup=[]

      for ent_in_group in entsToParse

        if( ent_in_group.is_a? Sketchup::Face )
          faces_were_found=true
          faces_in_group.push ent_in_group
        elsif ( ent_in_group.is_a? Sketchup::Group )
          groups_were_found=true
          groups_in_group.push ent_in_group
        elsif ( ent_in_group.is_a? Sketchup::ComponentInstance )
          groups_were_found=true
          groups_in_group.push ent_in_group
        else
          nonSupportedEntities_in_goup.push ent_in_group
        end

      end

      if groups_were_found

         if( $ParsingMode == "analyze" )
            $totalTreeGroupsInWorld+=1

         else

            if $consoleOutput
               #$tempstring+="\n"
               #puts( $tempstring )
               indent ( $file_level )
               $tempstring+="["+$file_level.to_s+"] "

               if ( ent.is_a? Sketchup::Group )
                  $tempstring+="[ GROUP "
                   $tempstring+=entid.to_s+" ]"
                   $tempstring+=($localmirror? "(!!!) " : " " )
                   groupname = ent.name
                   $tempstring+=" \"" + groupname +"\""

               elsif ( ent.is_a? Sketchup::ComponentInstance )
                   $tempstring+="[ COMPONENT "
                   $tempstring+=entid.to_s+" ]"
                   $tempstring+=( $localmirror? "(!!!) " : "  " )

                   defin=ent.definition
                   compdefname = defin.name
                   $tempstring+=" \"" + compdefname +"\""
                   end

             if( $ParsingMode == "outputbrushes" )
                 groupIDnum=$nextavailable_id
             else
                 groupIDnum=entid
                 end

                 $currentGroupID[ $file_level ] = groupIDnum

             if( $ParsingMode == "outputbrushes" )
                 
                 $grouptoHammerIDmapping[entid] = groupIDnum
                 #puts( entid.to_s + " = " + $grouptoHammerIDmapping[entid].to_s )
    
                 write_group( groupIDnum )
                 $nextavailable_id+=1

                 end


             #$tempstring+=", # within:" + groups_in_group.length.to_s
             $tempstring+=", mat: " + materialOfGroup
             $tempstring+=", contains (" + groups_in_group.length.to_s + ") objects:"
             if ( $ParsingMode == "outputbrushes" ) then $tempstring+=" --> { GROUP "+groupIDnum.to_s+" }" end
             puts( $tempstring )


             indent ( $file_level )
             #$tempstring+="["
             #puts( $tempstring )
             end


         end

         if groups_in_group.length < 2

         if( $ParsingMode == "outputbrushes" )
             $totalSinglyGroupedGroupInWorld +=1
            #if groups_in_group.length < 2
             #UI.messagebox "Exporter Warning: Group contains just one object"
             puts( "WARNING:  Group contains just one object. May create invalid brushes." )
           #else
         end

         end

      elsif faces_were_found

         if( $ParsingMode == "analyze" )
             $totalBrushesInWorld+=1
         elsif( $ParsingMode == "outputbrushes" )
             $tempstring+="["+$file_level.to_s+"] "

             if ( ent.is_a? Sketchup::Group )
                groupname= ent.name
                $tempstring+=" Group"
                $tempstring+=" "+entid.to_s
                $tempstring+=($localmirror? " (!!!)" : "" )
                $tempstring+=": \""+groupname+"\""
             elsif ( ent.is_a? Sketchup::ComponentInstance )
                $tempstring+=" Component"
                $tempstring+=" "+entid.to_s
                $tempstring+=($localmirror? " (!!!)" : "" )
                defin=ent.definition
                compdefname = defin.name
                $tempstring+=" def: (" + compdefname +")"
                if $su_version > 5.0    #not supported in previous versions
                  compinstancename = ent.name
                  $tempstring+=" \"" + compinstancename +"\""
                  end
                end

             $tempstring+=",   mat: " + materialOfGroup
             $tempstring+="   --> { SOLID BRUSH }"
             #if $currentxformismirrored[ $file_level ] then $tempstring+=" (%)," end
             if checkglobalmirror( $file_level ) then$tempstring+=" (!!!)" end

             if $consoleOutput then puts( $tempstring ) end

             write_solid( faces_in_group )

             end

        else

        #UI.messagebox "Unsupported Entities!"
        if $debugAlert then UI.messagebox "Unsupported Entities - This line shouldn't be Executing: line714" end
        puts( "Unsupported Entities - This Shouldn't be Executing: line714" )
      end

      # Go through the nested groups/comps
      for ent in groups_in_group

         if( $ParsingMode == "outputbrushes" )
         UpdatePercentComplete()
         end

         parse(ent)

      end


      # close out the console output
      if groups_were_found
         if( $ParsingMode != "analyze" )
           if $consoleOutput
             indent ( $file_level )
             $tempstring=$tempstring+" "
             puts( $tempstring )
           end
         end
      end

      $file_level = $file_level - 1


    elsif( ent.is_a? Sketchup::Face )
        if( $ParsingMode == "outputbrushes" )
            $totalOrphanedFacesInWorld+=1
            $tempstring="WARNING: {"+$file_level.to_s+"} "
            $tempstring+="UnGrouped (orphaned) Face "+entid.to_s
            puts( $tempstring )
        end

    else
        if( $ParsingMode == "outputbrushes" )
            $totalUnrecognizedInWorld+=1
            typse = ent.typename
            $tempstring="WARNING: {"+$file_level.to_s+"} "
            $tempstring+="Unsupported Entity: "+typse.to_s+" "+entid.to_s
            puts( $tempstring )
        end
    
    end #of if( ent.is_a? Sketchup::Group or ent.is_a? Sketchup::ComponentInstance )

    return true
end #of def parse( ent )
















def checkglobalmirror( lvl )

#  puts lvl
#  indent ( lvl )

  totalmirrorsinstack=0
  lvl.times do |i|

#    puts i
    i=i+1
    if $currentxformismirrored[ i ]
#       $tempstring+= i.to_s+"+ "
       totalmirrorsinstack+=1
#    else
#       $tempstring+= i.to_s+"- "
    end

#    next if i % 2 == 0
  end

#    puts( $tempstring )

  if totalmirrorsinstack % 2 == 0
     return false
  else
     return true
  end

end


def therearemirrors( lvl )

#  puts lvl
#  indent ( lvl )

  totalmirrorsinstack=0
  lvl.times do |i|

#    puts i
    i=i+1
    if $currentxformismirrored[ i ]
#       $tempstring+= i.to_s+"+ "
       totalmirrorsinstack+=1
#    else
#       $tempstring+= i.to_s+"- "
    end

#    next if i % 2 == 0
  end

#    puts( $tempstring )

  if totalmirrorsinstack > 0
     return true
  else
     return false
  end

end














































def write_group( idnum )
  if $debugOutput then $export_file.puts( "\n\t//"+ $tempstring +"\n" ) end

  $export_file.puts( "\tgroup" )
  $export_file.puts( "\t{" )

  $export_file.puts( "\t\t\"id\" \"" + idnum.to_s + "\"" )
  $export_file.puts( "\t\teditor" )
  $export_file.puts( "\t\t\{" )
  $export_file.puts( "\t\t\t\"color\" \"0 180 193\"" )

   if $file_level > 1
#   Put the last discovered Group ID
    onetoget =  $file_level -1
    groupidnum=$currentGroupID[ onetoget ]
#    puts(  "*** Inside GroupID[" + onetoget.to_s +  "] : " + $currentGroupID[ onetoget ].to_s)
    $export_file.puts(  "\t\t\t\"groupid\" \"" + groupidnum.to_s + "\"" )
   end

  #Add current group ID info here

  $export_file.puts( "\t\t\t\"visgroupshown\" \"1\"" )
  $export_file.puts( "\t\t\t\"visgroupautoshown\" \"1\"" )
  $export_file.puts( "\t\t\}" )

  $export_file.puts( "\t\}" )

end


# ================================ Export a Solid brush
def write_solid( faces )

    if $debugOutput then $export_file.puts( "\n\t//"+ $tempstring +"\n" ) end
    
    if faces.length < 4
       puts( "\n   WARNING: Invalid brush detected. (less than 4 faces) " )
       if $debugOutput then $export_file.puts( "\n   WARNING: Invalid brush detected. (less than 4 faces) " ) end
       #return false
       end


# takes four or more faces and makes a solid brush. Should be convex. Ideally, there should be code to check convexity...

#    if therearemirrors( $file_level )
#       puts( "slow" )
#    else
#       puts( "fast" )
#    end


#Write solid header, unique ID


  $export_file.puts( "\tsolid" )
  $export_file.puts( "\t{" )
  $export_file.puts( "\t\t\"id\" \"" + $nextavailable_id.to_s + "\"" )
  $nextavailable_id+=1

  #Write out faces
  i = 0
  faces.each do |that|
    brushside = faces[i]

    if $brush_mode=="fast"
    #if therearemirrors( $file_level )
       writefacefast( brushside )
       #writefacefast( brushside )
    else
       writefaceslow( brushside )
       #writefacefast( brushside )
    end

    i=i+1
  end

  $export_file.puts( "\t\teditor" )
  $export_file.puts( "\t\t\{" )
  $export_file.puts( "\t\t\t\"color\" \"0 180 193\"" )

# GroupID info goes here

   if $file_level > 1
#   Put the last discovered Group ID
    onetoget =  $file_level -1
    groupidnum=$currentGroupID[ onetoget ]
#    puts(  "*** Inside GroupID[" + onetoget.to_s +  "] : " + $currentGroupID[ onetoget ].to_s)
    $export_file.puts(  "\t\t\t\"groupid\" \"" + groupidnum.to_s + "\"" )
   end

  $export_file.puts( "\t\t\t\"visgroupshown\" \"1\"" )
  $export_file.puts( "\t\t\t\"visgroupautoshown\" \"1\"" )
  $export_file.puts( "\t\t\}" )

  #Add current group ID info here

  #Close out the solid entry
  $export_file.puts( "\t}" )

  return true

end




# THe fast new method
def writefacefast( facein )
    entid=facein.entityID
    $file_level = $file_level + 1

    #Process Material override to hierarchy level
    materialnameof_face=get_material_name( facein )
    materialname=materialnameof_face

    if( materialnameof_face == "_unspecified_" )
        $currentmaterial_name[ $file_level ]=$currentmaterial_name[ $file_level-1 ]
        materialname=$currentmaterial_name[ $file_level ]
        end

    if $debugOutput
       $export_file.puts( ""  )
       indent ( $file_level )
       $tempstring=$tempstring+"\t// ["+$file_level.to_s+"] FACE "+entid.to_s+", mat: " + materialnameof_face

       if( materialnameof_face == "_unspecified_" )
           $tempstring=$tempstring + " ( using defaultmaterial[" + $file_level.to_s + "]: " + materialname +" )"
           end

       $export_file.puts( $tempstring  )
       end

    # Check the material for a valid texture. returns nil if none, in which case a default texture size is used.
    uv_width=512
    uv_height=512

    matTexStatus=false
    if matTexStatus
      puts ("Texture Width:  "+materialname.texture.width.to_s )
      puts ("Texture Height: "+materialname.texture.height.to_s )
       uv_width=materialname.texture.width
       uv_height=materialname.texture.height
       end

    $export_file.puts( "\t\tside\n\t\t{\n")
    $export_file.puts( "\t\t\t\"id\" \"" + $nextavailable_id.to_s + "\"\n" )
    $nextavailable_id+=1


    #Start building Plane output line
    tempstring = "\t\t\t\"plane\" \""

    localverts=facein.vertices

    #If the current xfrom stack is mirrored
    if checkglobalmirror( $file_level )

       #puts( "Reversing")
       #Format the brush plane information for output
       cnt = 0
       0.upto(2) do |that|
           localCoord = Geom::Point3d.new( localverts[cnt].position.x, localverts[cnt].position.y, localverts[cnt].position.z )

           #transform to worldspace
           global = cumulatexform( localCoord )

           #$export_file.puts( cnt )
           tmpx = sprintf( "%.1f",localCoord.x.to_f.to_inch )
           tmpy = sprintf( "%.1f",localCoord.y.to_f.to_inch )
           tmpz = sprintf( "%.1f",localCoord.z.to_f.to_inch )
           tempstring += "(" + tmpx + " " + tmpy + " " + tmpz + ")"

           if(cnt == 2)
               tempstring += "\""
           else
               tempstring += " "
               end

           cnt = cnt + 1
           end

    #If the current xfrom stack is normal (non-mirrored)
    else

        #puts( "Normal")
        #Format the brush plane information for output
        cnt = 2
        0.upto(2) do |that|
           localCoord = Geom::Point3d.new( localverts[cnt].position.x, localverts[cnt].position.y, localverts[cnt].position.z )

           #transform to worldspace
           global = cumulatexform( localCoord )

           #$export_file.puts( cnt )
           tmpx = sprintf( "%.1f",localCoord.x.to_f.to_inch )
           tmpy = sprintf( "%.1f",localCoord.y.to_f.to_inch )
           tmpz = sprintf( "%.1f",localCoord.z.to_f.to_inch )
           tempstring += "(" + tmpx + " " + tmpy + " " + tmpz + ")"

           if(cnt == 0)
               tempstring += "\""
           else
               tempstring += " "
               end

           cnt = cnt - 1
           end

       end

    #Write the Plane info to file
    $export_file.puts( tempstring )

    if $uv_mode == 1
       UVcalcbasic( facein )

    elsif $uv_mode == 2
       #UVcalcAdvanced( facein )
       if $debugOutput then $export_file.puts( "\t\t\t\//(Advanced UV mode TBA )" ) end
    else
       # don't output any UV data
       if $debugOutput then $export_file.puts( "\t\t\t\//(No UV info)" ) end
       end


    $export_file.puts( "\t\t\t\"material\" \"" + materialname.upcase + "\"" )

    $export_file.puts( "\t\t\t\"rotation\" \"0\"" )
    $export_file.puts( "\t\t\t\"lightmapscale\" \"16\"" )
    $export_file.puts( "\t\t\t\"smoothing_groups\" \"0\"" )

    $file_level = $file_level - 1

    $export_file.puts( "\t\t}\n")

    return true
    
    end # of function (writefacefast)










# UVs quick, based on normal.
def UVcalcbasic( facein )

    origin_local=Geom::Point3d.new  #(0,0,0)
    #if $debugOutput then $export_file.puts( "\t\t\t// origin before "+origin_local.to_s ) end

    facenormal_local = facein.normal #normal of the face in local space
    if $debugOutput then $export_file.puts( "\t\t\t// face normal(local)"+facenormal_local.to_s ) end

    #Need to convert to a point because of SU vector xform difficulties
    facenormal_local_point = Geom::Point3d.new( facenormal_local.x, facenormal_local.y, facenormal_local.z )
    #if $debugOutput then $export_file.puts( "\t\t\t// face normal(local) as point "+facenormal_local_point.to_s ) end

    #transform to worldspace
    origin_xformed = cumulatexform( origin_local )
    #if $debugOutput then $export_file.puts( "\n\t\t\t// origin XFORMED "+origin_xformed.to_s ) end

    facenormal_xformed = cumulatexform( facenormal_local )
    if $debugOutput then $export_file.puts( "\t\t\t// face normal XFORMED "+facenormal_xformed.to_s ) end

    vec = Geom::Vector3d.new( facenormal_xformed.x, facenormal_xformed.y, facenormal_xformed.z )
    #if $debugOutput then $export_file.puts( "\t\t\t// convert to vector   "+vec.to_s ) end

    #faceglobalnormal = newpoint - origin_xformed

    uvaxis = Geom::Transformation.new( origin_xformed, facenormal_xformed )

    $tempstring = "\t\t\t\"uaxis\" \""

        uv_vectorx = sprintf( "%.3f",uvaxis.xaxis.x )
        uv_vectory = sprintf( "%.3f",uvaxis.xaxis.y )
        uv_vectorz = sprintf( "%.3f",uvaxis.xaxis.z )
        $tempstring += "[" + uv_vectorx + " " + uv_vectory + " " + uv_vectorz + " 0] 0.25\""

    
    $export_file.puts( $tempstring )

    $tempstring = "\t\t\t\"vaxis\" \""
    
        uv_vectorx = sprintf( "%.3f",uvaxis.yaxis.x )
        uv_vectory = sprintf( "%.3f",uvaxis.yaxis.y )
        uv_vectorz = sprintf( "%.3f",(-1*uvaxis.yaxis.z) )  #This one seems to need reversing...
        $tempstring += "[" + uv_vectorx + " " + uv_vectory + " " + uv_vectorz + " 0] 0.25\""

    
    $export_file.puts( $tempstring )


end



      #transform to worldspace
#      fl=$file_level-1
#      t = Geom::Transformation.new $currentxform[fl]
#      localCoord.transform! t
#      global = Geom::Point3d.new
#      global = localCoord


#      lev=$file_level
#      1.upto( lev ) do |that|
#        trans = Geom::Transformation.clone $currentxform[ lev ]
#        localCoord.transform!( trans )
#      lev+=1
#      end


#point = Geom::Point3d.new 10,20,30
#t = Geom::Transformation.new point

#wector = Geom::Vector3d.new 10,20,30
#t = Geom::Transformation.translation wector

#point = Geom::Point3d.new 0,0,0
#vectord = Geom::Vector3d.new 0,0,1
#angle = 2
#t = Geom::Transformation.rotation point, vectord, angle

#t = Geom::Transformation.scaling 10 #WORKS

#x = t.origin
#UI.messagebox x

#UI.messagebox $currentxform[ $file_level ]
#asd = Geom::Vector3d.new 1000,2000,3000
#UI.messagebox t.xaxis

#	trans = e.transformation.to_a
#	x = trans[12]
#	y = trans[13]
#	z = trans[14]





#Add up the current transform chain by applying each the input
def cumulatexform( point3Din )

    global = Geom::Point3d.new
    coord=point3Din
    xtotal = $file_level - 1  # The total number of levels to be done
    xcnt = xtotal

    1.upto(xtotal) do |that|

      t = Geom::Transformation.new $currentxform[xcnt]
      coord.transform! t

      xcnt = xcnt - 1

    end

    global = coord

    return global
end

def flipwinding()
    if $localmirror
       $localmirror=false
    else
       $localmirror=true
    end
end





# ============================================================ Export a Face as a brush Side
def writefaceslow( facein )

#    puts( "SLOW")
    entid=facein.entityID

    $file_level = $file_level + 1

    faceverts=facein.vertices

    #Process Material override to hierarchy level
    materialnameof_face=get_material_name( facein )
    materialname=materialnameof_face

    if( materialnameof_face == "_unspecified_" )
        $currentmaterial_name[ $file_level ]=$currentmaterial_name[ $file_level-1 ]
        materialname=$currentmaterial_name[ $file_level ]
    end

    #Material Debug Info
    if $debugOutput
       $tempstring=""
       indent ( $file_level )
       $tempstring=$tempstring+"//\t["+$file_level.to_s+"] *** FACE "+entid.to_s+", mat: " + materialnameof_face

       if( materialnameof_face == "_unspecified_" )
         $tempstring=$tempstring + " (using cm" + $file_level.to_s + ": " + materialname +")"
       end

       $export_file.puts( $tempstring  )
    end

    # Check the material for a valid texture.
    matStatus = facein.material

#    if matTextureStatus
#       puts facein.material.texture.height.to_s
#       puts facein.material.texture.width.to_s
#    end

    $export_file.puts( "\t\tside\n\t\t{\n")
    $export_file.puts( "\t\t\t\"id\" \"" + $nextavailable_id.to_s + "\"\n" )
    $nextavailable_id+=1

    tempstring = "\t\t\t\"plane\" \""
    #Start building Plane output line

    localverts = facein.vertices
    globalverts=[]

    globalnormal=Geom::Vector3d.new
    orig=Geom::Point3d.new


#Sketchup.active_model.layers.add("0vmftemp")
#Sketchup.active_model.active_layer = ("0vmftemp")

#SketchUp keeps all face info in localspace and uses cumulative hierarchical transforms on grouped objects
#Hammer wants brushes in world coordinates, So this experiment

    modelentities = $tempmodel.active_entities

    # Add empty groups to the entities in the model
    tempFacegroup=modelentities.add_group
    tempUVgroup=modelentities.add_group

    #define the entities within the group
    tempentities=tempFacegroup.entities
    tempUVentities=tempUVgroup.entities

    # Add a face to within the temp group
    face = tempentities.add_face localverts #The whole face
    #face = tempentities.add_face( localverts[0], localverts[1], localverts[2] ) #just a triangle
#UI.messagebox "Create copy of face (in local space)"

    # Add a proxy that wil retain displacement for the UV origin after exploding the tempUVgroup
    # using an image object because cpoint doesn't seem to want to work without errors...
    #uvorigin = tempUVentities.add_image("C:\\1REFTEX.jpg", orig, 1 ) #the

    reftex = File.join( $pluginsfolder, "hammertime", "1REFTEX.jpg")
    uvorigin = tempUVentities.add_image( reftex, orig, 1 ) #the

#UI.messagebox "Temp image in local space"

    $templocalnormal = face.normal

    ptz=Geom::Point3d.new( $templocalnormal.x, $templocalnormal.y, $templocalnormal.z )

    # The local normal, or texture W axis
    w_axisline = tempUVentities.add_line(ptz, orig)
#UI.messagebox "This line is the local normal, z axis or texture W axis"

    # Use the transform trick to generate the other axes the lazy way.
    uvwtransform = Geom::Transformation.new( orig, $templocalnormal )
    #$export_file.puts( "Local UV x axis: " + uvwtransform.xaxis.to_s )
    #$export_file.puts( "Local UV y axis: " + uvwtransform.yaxis.to_s )

    ptx=Geom::Point3d.new( uvwtransform.xaxis.x, uvwtransform.xaxis.y, uvwtransform.xaxis.z )
    pty=Geom::Point3d.new( uvwtransform.yaxis.x, uvwtransform.yaxis.y, uvwtransform.yaxis.z )

    u_axisline = tempUVentities.add_line(ptx, orig)
#UI.messagebox "This is the U axis,"

    v_axisline = tempUVentities.add_line(pty, orig)
#UI.messagebox "This is the V axis,"

    u_planeface = tempUVentities.add_face( orig, ptz, pty ) #a plane normal to the u axis
    uplane = u_planeface.plane
#UI.messagebox "Create U (texture X) plane..."

    v_planeface = tempUVentities.add_face( ptx, ptz, orig ) #a plane normal to the v axis
    vplane = v_planeface.plane
#UI.messagebox "Create V (texture Y) plane..."

    w_planeface = tempUVentities.add_face( pty, ptx, orig ) #a plane normal to the w axis
    uvverts = w_planeface.vertices
#UI.messagebox "Create W (texture z) plane..."


    #Concatenate the cumulative current transform chain by applying each to grouped temp face
    xtotal = $file_level - 1
    xcnt = xtotal
    1.upto(xtotal) do |that|
      tempFacegroup.transform!( $currentxform[ xcnt ] )
      tempUVgroup.transform!( $currentxform[ xcnt ] )

      xcnt = xcnt - 1
    end
#UI.messagebox "xform both"

    # Explode the facegroup. The face now has the collapsed position of the brush plane
    tempfaceents=tempFacegroup.explode
#    if $debugAlert then UI.messagebox "temp face exploded" end

    #Extract the global vertex points
    i=0
    0.upto(tempfaceents.length - 1) do |that|

          if( tempfaceents[i].is_a? Sketchup::Face )
           globalverts = tempfaceents[i].vertices
           globalnormal = tempfaceents[i].normal
          end
    i = i + 1
    end

    #filter out faces and edges,
    #I need this because edge uses are causing me trouble...
    finalfaceentstemp=[]
    tempfaceents.each do |element|
      if( element.typename == "Face" )
          finalfaceentstemp.push element
      end
    end

    tempfaceents.each do |element|
      if( element.typename == "Edge" )
          finalfaceentstemp.push element
      end
    end

    # Format the brush plane information for output
    cnt = 2
    0.upto(2) do |that|
      vert = Geom::Point3d.new( globalverts[cnt].position.x, globalverts[cnt].position.y, globalverts[cnt].position.z )

      tmpx = sprintf( "%.1f",vert.x.to_f.to_inch )
      tmpy = sprintf( "%.1f",vert.y.to_f.to_inch )
      tmpz = sprintf( "%.1f",vert.z.to_f.to_inch )
      tempstring += "(" + tmpx + " " + tmpy + " " + tmpz + ")"

      if(cnt == 0)
         tempstring += "\""
      else
         tempstring += " "
      end

      cnt = cnt - 1
    end

    #Delete the temporary vertex points for the brush face
    finalfaceentstemp.each do |e|
     if not e.deleted?
      e.erase!
     end
    end
#UI.messagebox "Delete the brush plane local face"

    $export_file.puts( tempstring )
    #Write the Plane info to file

    $export_file.puts( "\t\t\t\"material\" \"" + materialname.upcase + "\"" )




    #**************************
    # Begin processing UV info
    #**************************

    tempUVents=tempUVgroup.explode
#UI.messagebox "explode UV proxy to collapse cumulative transforms"

    uvw_proxy_ents=[]
    u_planeverts=[]
    v_planeverts=[]
    w_planeverts=[]

    # filter out the temp UV package, capture the vertices
    tempUVents.each do |ent|
      if( ent.typename == "Image" )
          diplacementPoint = ent.origin
          #$export_file.puts( "The xformed UV origin: " + diplacementPoint.to_s )

          vec = Geom::Vector3d.new(diplacementPoint.x, diplacementPoint.y, diplacementPoint.z)

          $displacementTransform = Geom::Transformation.translation(vec)

          uvw_proxy_ents.push(ent)
      end
      if( ent == u_planeface )
#UI.messagebox "found the u plane"
              uplane = u_planeface.plane
              u_planeverts = u_planeface.vertices
      end
      if( ent == v_planeface )
#UI.messagebox "found the v plane"
              vplane = v_planeface.plane
              v_planeverts = v_planeface.vertices
      end
      if( ent == w_planeface )
#UI.messagebox "found the w plane"
          w_planeverts = w_planeface.vertices
      end
      if( ent.typename == "Face" )
          # collect them all, faces, that is.
          uvw_proxy_ents.push(ent)
      end
      if( ent.typename == "Edge" )
          uvw_proxy_ents.push(ent)
      end
    end


    tempUfacegroup=modelentities.add_group
    tempfaceentities=tempUfacegroup.entities

    tempVfacegroup=modelentities.add_group
    tempVfaceentities=tempVfacegroup.entities

    uplaneface = tempfaceentities.add_face u_planeverts #a plane normal to the u axis
#UI.messagebox "duplicate u plane"

    vplaneface = tempVfaceentities.add_face v_planeverts #a plane normal to the v axis
#UI.messagebox "duplicate v plane"

ss = Sketchup.active_model.selection





   # move the proxy back to the origin, and adjust for displacement
   newProxygroup = modelentities.add_group(uvw_proxy_ents)

#ss.add(newProxygroup)
#UI.messagebox "Regroup UV proxy"
#ss.clear




   $displacementTransform.invert!
   newProxygroup.transform!( $displacementTransform )
#UI.messagebox "move UV proxy back to the origin"

#ss.clear
#ss.add(tempUfacegroup)
#UI.messagebox "explode the new u plane"
   intplanefaceents=tempUfacegroup.explode
#UI.messagebox "exploded the new u plane"

   u_plane_ents=[]
   intplanefaceents.each do |ent|

       if( ent.typename == "Face" )
          uplaneface=ent
#ss.clear
#ss.add(uplaneface)
#UI.messagebox "U Face"

          u_plane_ents.push(ent)
       end
       if( ent.typename == "Edge" )
         u_plane_ents.push(ent)
       end

   end

kringplane=uplaneface.plane

    tempUfacegroup=modelentities.add_group(u_plane_ents)



#explode the new V plane
   intplanefaceents=tempVfacegroup.explode
#UI.messagebox "exploded the new u plane"

   v_plane_ents=[]
   intplanefaceents.each do |ent|

       if( ent.typename == "Face" )
          vplaneface=ent
          v_plane_ents.push(ent)
       end
       if( ent.typename == "Edge" )
         v_plane_ents.push(ent)
       end

   end

kringVplane=vplaneface.plane

    tempVfacegroup=modelentities.add_group(v_plane_ents)


#ss.add(newProxygroup)
   pUVentsfixed=[]
   pUVentsfixed=newProxygroup.explode
#UI.messagebox "explode UV proxy in its final coordinate space"

   #get the UV lengths
   #apply scaling factor to match source engine standards
   scale_U=u_axisline.length * 1
   scale_V=v_axisline.length * 1
   #scale_V=v_axisline.length * $global_texture_scale_factor

#ss.add(u_axisline)
#UI.messagebox( "u axis length: "+scale_U.to_s )
#UI.messagebox( "v axis length: "+scale_V.to_s )
#ss.clear


   alertmessage0=sprintf( "%.3f",scale_U.to_f )
   alertmessage1=sprintf( "%.3f",scale_V.to_f )
   alertmessage2="Uscale: "+ alertmessage0 + "   Vscale: "+ alertmessage1

#   if $debugAlert then UI.messagebox alertmessage2 end

   # empty out the uv proxy package array for refiltering
   #uvw_proxy_ents.clear
#UI.messagebox "next      regroup to idsolate"

   #newProxygrouptwo = modelentities.add_group(tempUVentsfixed)
#UI.messagebox "regroup to idsolate"







   # Extract the UV axes from the w plane in the form of vertex array, delete the proxy entities
    pUVentsfixed.each do |ent|
       if( ent.typename == "Image" )
          uvw_proxy_ents.push(ent)
       end
       if( ent == w_planeface )
          w_planeverts = ent.vertices
       end
       if( ent.typename == "Face" )
          uvw_proxy_ents.push(ent)
       end
       if( ent.typename == "Edge" )
         uvw_proxy_ents.push(ent)
       end
    end


   strt = Geom::Point3d.new( 0,0,0 )
   intUaxis = Geom::Point3d.new( 0,0,0 )
   intVaxis = Geom::Point3d.new( 0,0,0 )

   #add a line to do intersection
   intUaxis = Geom.intersect_line_plane(u_axisline.line, kringplane)
   intVaxis = Geom.intersect_line_plane(v_axisline.line, kringVplane)
#UI.messagebox(". "+intUaxis.to_s, MB_MULTILINE , "intersection")
#UI.messagebox(". "+intVaxis.to_s, MB_MULTILINE , "intersection")

   uoffset=0.00
   if (intUaxis == strt)
      uoffset=0.00
   else

#udisplacementline = modelentities.add_line(strt, intUaxis)
#UI.messagebox("try adding line")
#udisplacementline.erase!


       if ( intUaxis[0] >= 0 )
          uoffset =(-1 * ( strt.distance intUaxis ) )
       else
          uoffset =(1 * ( strt.distance intUaxis ) )
       end

      #uoffset=(udisplacementline.length * -1)
      #needs to be negative? I guess axis moves brush relative to texture space...
   end

   voffset=0.00
   if (intVaxis == strt)
      voffset=0.00
   else

#UI.messagebox("vy: "+voffset.to_s )
       if ( intVaxis[2] >= 0 )
          voffset =(-1 * (  strt.distance intVaxis  )   )
       else
          voffset =(1 * (  strt.distance intVaxis  )   )
       end

   end


#Hammer scales by altering the scaling factor and the displacement. Try not to scale the axes vector?

#UI.messagebox "Draw Line "+uoffset.to_s

#      uoffset=0.000000


   #u_plane_ents.push(udisplacementline)

    # Write out the coordinates of the UVW axis

    $tempstring = "\t\t\t\"uaxis\" \""

        uxtemp=w_planeverts[0].position.x
        uxtempx=uxtemp/scale_U
        #Careful of divide by 0
#UI.messagebox uxtemp.to_s+"/"+scale_U.to_s+" ="+uxtempx.to_s
        tempstringx = sprintf( "%.3f",uxtempx.to_f.to_inch )

        uytemp=w_planeverts[0].position.y
#        uytempy=(uytemp/scale_V)
#UI.messagebox materialname+" Uscale "+uytempy.to_s
        tempstringy = sprintf( "%.3f",uytemp.to_f.to_inch )

        uztemp=w_planeverts[0].position.z
        tempstringz = sprintf( "%.3f",uztemp.to_f.to_inch )

        uScaletemp=scale_U
        uoffsetsc=uoffset/scale_U
        tempstringud = sprintf( "%.3f",(uoffsetsc.to_f.to_inch) )

        tempstringS = sprintf( "%.3f",(uScaletemp).to_f )
        $tempstring += "[" + tempstringx + " " + tempstringy + " " + tempstringz + " " + tempstringud + "] "+ tempstringS +"\""


    $export_file.puts( $tempstring )

    $tempstring = "\t\t\t\"vaxis\" \""

        uxtemp=w_planeverts[1].position.x * -1

        tempstringx = sprintf( "%.3f",uxtemp.to_f.to_inch )

        uytemp=w_planeverts[1].position.y * -1
        tempstringy = sprintf( "%.3f",uytemp.to_f.to_inch )

        uztemp=w_planeverts[1].position.z * -1
        uztempz=uztemp/scale_V
        tempstringz = sprintf( "%.3f",uztempz.to_f.to_inch )
#UI.messagebox materialname+" Vscale "+uztempz.to_s

        vScaletemp=scale_V
        voffsetsc=(-1*voffset)/scale_V
        tempstringvd = sprintf( "%.3f",(voffsetsc.to_f.to_inch) )


        tempstringS = sprintf( "%.3f",vScaletemp.to_f )

        $tempstring += "[" + tempstringx + " " + tempstringy + " " + tempstringz + " " + tempstringvd + "] "+ tempstringS +"\""

    
    $export_file.puts( $tempstring )
#UI.messagebox "write to file complete..."

    uvw_proxy_ents.each do |ent|
     if not ent.deleted?
       ent.erase!
     end
    end
#UI.messagebox "erase UVW axis Proxy complete..."

   intplanefaceents=tempUfacegroup.explode
#UI.messagebox "exploded the new u plane"

   u_plane_ents=[]
   intplanefaceents.each do |ent|

       if( ent.typename == "Face" )
          uplaneface=ent
          u_plane_ents.push(ent)
       end
       if( ent.typename == "Edge" )
         u_plane_ents.push(ent)
       end

   end

    i=0
    0.upto(u_plane_ents.length - 1) do |that|

          if( u_plane_ents[i].is_a? Sketchup::Face or u_plane_ents[i].is_a? Sketchup::Edge )
             if not u_plane_ents[i].deleted?
               u_plane_ents[i].erase!
             end
          end
    i = i + 1
    end
#UI.messagebox "erase UV displace Proxy complete"

   intplanefaceents=tempVfacegroup.explode
#UI.messagebox "exploded the new u plane"

   v_plane_ents=[]
   intplanefaceents.each do |ent|

       if( ent.typename == "Face" )
          uplaneface=ent
          v_plane_ents.push(ent)
       end
       if( ent.typename == "Edge" )
         v_plane_ents.push(ent)
       end

   end

    i=0
    0.upto(v_plane_ents.length - 1) do |that|

          if( v_plane_ents[i].is_a? Sketchup::Face or v_plane_ents[i].is_a? Sketchup::Edge )
             if not v_plane_ents[i].deleted?
               v_plane_ents[i].erase!
             end
          end
    i = i + 1
    end
#UI.messagebox "erase UV displace Proxy complete"





if $uvoverride


# Global auto align UVs
    tempnormal = globalnormal
    normtrans = Geom::Transformation.new( orig, tempnormal )

    uvwtransform = normtrans

    normx = normtrans.xaxis

    $tempstring = "\t\t\t\"uaxis\" \""

        uxtemp=uvwtransform.xaxis.x
        tempstringx = sprintf( "%.3f",uxtemp )
        uytemp=uvwtransform.xaxis.y
        tempstringy = sprintf( "%.3f",uytemp )
        uztemp=uvwtransform.xaxis.z
        tempstringz = sprintf( "%.3f",uztemp )
        $tempstring += "[" + tempstringx + " " + tempstringy + " " + tempstringz + " 0] 0.25\""


    $export_file.puts( $tempstring )

    $tempstring = "\t\t\t\"vaxis\" \""

        vxtemp=uvwtransform.yaxis.x
        tempstringx = sprintf( "%.3f",vxtemp )
        vxtemp=uvwtransform.yaxis.y * -1
        tempstringy = sprintf( "%.3f",vxtemp )
        vxtemp=uvwtransform.yaxis.z * -1
        tempstringz = sprintf( "%.3f",vxtemp )
        $tempstring += "[" + tempstringx + " " + tempstringy + " " + tempstringz + " 0] 0.25\""


    $export_file.puts( $tempstring )



    if( normx == 0)
       ang = 90
    else
       ang = 90
       #ang = Math.atan2(normy, normx)
       #ang = ( ( Math.atan2(normy, normx) )/ 0.017453293 )
       #ang = (normy/normx)
    end

    #export_file.puts( "Y over X: " + ang.to_s )



    # [ texture shift ] Texture Scale factor

    uam=(-1 * Math.sin(ang) )
    ubm=Math.cos(ang)

end
# of global to face axis auto UVW

    $export_file.puts( "\t\t\t\"rotation\" \"0\"" )
    $export_file.puts( "\t\t\t\"lightmapscale\" \"16\"" )
    $export_file.puts( "\t\t\t\"smoothing_groups\" \"0\"" )

    $file_level = $file_level - 1

    $export_file.puts( "\t\t}\n")

end

# ================================ Get a Material from an object

# The outer shell should be a dark grey with white grid dev/dev_measuregeneric01b
# default geometry should be lighter grey dev/graygrid

def get_material_name( obj )
  if (obj.material == nil)
    col = "_unspecified_"
  else
    col = obj.material.display_name
  end
  return col
end

# ================================ debug indenting

def indent ( indentions )
    $tempstring=""
    1.upto(indentions) do |that|
        $tempstring=$tempstring+"   "
    end
end

def indentconsole ( indentions )
    $tempstring=""
    1.upto(indentions) do |that|
        $tempstring=$tempstring+" "
    end
end


def writeBoundingBox( boundingBoxReceived )
    puts( "To Do: Write Bounding Box" )

end


# ============================================================ Warn if Hammer Max boundaries exceeded
# Largest allowed map without 3D skybox is 32768 units. 16384 max unit depth.

def ValidHammerMaxBounds( boundingBoxIn )
  validDims=true
  maxDim = 16384
  bbmin=Geom::Point3d.new boundingBoxIn.min
  bbmax=Geom::Point3d.new boundingBoxIn.max

#if $debugAlert
#  UI.messagebox bbmin
#  UI.messagebox bbmax
#end

  $tempstring="Error: SketchUp model exceeds the dimensional limits of Hammer and Source:\n"

  if ( bbmax.x > maxDim )
    $tempstring=$tempstring+" - Positive Red (+X)\n"
    validDims=false
  end
  if ( bbmax.y > maxDim )
    $tempstring=$tempstring+" - Positive Green (+Y)\n"
    validDims=false
  end
  if ( bbmax.z > maxDim )
    $tempstring=$tempstring+" - Positive Blue (+Z)\n"
    validDims=false
  end

  if ( bbmin.x.abs > maxDim )
    $tempstring=$tempstring+" - Negative Red (-X)\n"
    validDims=false
  end
  if ( bbmin.y.abs > maxDim )
    $tempstring=$tempstring+" - Negative Green (-Y)\n"
    validDims=false
  end
  if ( bbmin.z.abs > maxDim )
    $tempstring=$tempstring+" - Negative Blue (-Z)\n"
    validDims=false
  end

  $tempstring+="\nWays to fix it:\n1. Make the map smaller.\n2. Use a 3D skybox for distant geometry.\n3. Use a 2D skybox instead of geometry."

  if ( validDims == false)
     UI.messagebox $tempstring
  end

  return validDims

end



def modelisempty( )
    lmodel = Sketchup.active_model
    lentities = lmodel.entities

    if lentities.length == 0
       return true
    else
       return false
    end
end



# ============================================================ write out material
def cascadematerial( materialnamein )
end


def countEntity( ent, entName )
    entid=ent.entityID

    if entName == "light" then $atLeastOne_Light_Found=true end
    if entName == "light_environment" then $atLeastOne_Light_Found=true end
    if entName == "info_player_start" then $atLeastOne_PlayerStart_Found=true end
    end





def writeEntity( ent, entName )
    entid=ent.entityID

    $export_file.puts( "entity" )
    $export_file.puts( "{" )
    $export_file.puts( "\t\"id\" \"" + ent.entityID.to_s + "\"" )
    $export_file.puts( "\t\"classname\" \""+entName+"\"" )

    if entName == "light"
       $atLeastOne_Light_Found=true

       $export_file.puts( "\t\"_light\" \"255 255 255 500\"" )
       $export_file.puts( "\t\"_lightHDR\" \"-1 -1 -1 1\"" )
       $export_file.puts( "\t\"_lightscaleHDR\" \"1\"" )
       $export_file.puts( "\t\"_quadratic_attn\" \"1\"" )
       end

    if entName == "light_environment"
       $atLeastOne_Light_Found=true

       $export_file.puts( "\t\"_ambient\" \"64 64 64 200\"" )
       $export_file.puts( "\t\"_ambientHDR\" \"-1 -1 -1 1\"" )
       $export_file.puts( "\t\"_AmbientScaleHDR\" \"1\"" )
       $export_file.puts( "\t\"_light\" \"255 255 255 500\"" )
       $export_file.puts( "\t\"_lightHDR\" \"-1 -1 -1 1\"" )
       $export_file.puts( "\t\"_lightscaleHDR\" \"1\"" )
       $export_file.puts( "\t\"pitch\" \"-90\"" )
       $export_file.puts( "\t\"SunSpreadAngle\" \"5\"" )
       end

    if entName == "info_player_start"
       $atLeastOne_PlayerStart_Found=true
       end
       
    $export_file.puts( "\t\"angles\" \"0 0 0\"" )

    trans = ent.transformation
    pt=trans.origin

    #Concatenate the cumulative current transform chain by applying each to grouped temp face
    xtotal = $file_level - 1
    xcnt = xtotal

    1.upto(xtotal) do |that|
       pt.transform!( $currentxform[ xcnt ] )
       xcnt = xcnt - 1
       end

    ox = sprintf( "%.3f",pt.x.to_i )
    oy = sprintf( "%.3f",pt.y.to_i )
    oz = sprintf( "%.3f",pt.z.to_i )

    if $debugOutput
#      $export_file.puts( "x: " +pt.x.to_i.to_s )
#      $export_file.puts( "y: " +pt.y.to_i.to_s )
#      $export_file.puts( "z: " +pt.z.to_i.to_s )
       end

    $export_file.puts( "\t\"origin\" \""+ox+" "+oy+" "+oz+ "\"" )

    $export_file.puts( "\teditor" )
    $export_file.puts( "\t{" )
    $export_file.puts( "\t\t\"color\" \"0 255 0\"" )

# GroupID info goes here

   if $file_level > 1

#   Put the last discovered Group ID
    onetoget =  $file_level -1
    groupidnum=$currentGroupID[ onetoget ]

    groupID_inhammer = $grouptoHammerIDmapping[groupidnum]

#    puts(  "*** Inside GroupID[" + onetoget.to_s +  "] : " + $currentGroupID[ onetoget ].to_s)
    $export_file.puts(  "\t\t\"groupid\" \"" + groupID_inhammer.to_s + "\"" )
   end

    $export_file.puts( "\t\t\"visgroupshown\" \"1\"" )
    $export_file.puts( "\t\t\"visgroupautoshown\" \"1\"" )
    $export_file.puts( "\t\t\"logicalpos\" \"[0 0]\"" )
    $export_file.puts( "\t}" )
    $export_file.puts( "}" )

    end






#Write out a transformation matrix to the outputfile, for debugging only.
def spitoutXform( t )

    $export_file.puts( "" )
    $export_file.puts( t[0].to_s+" "+t[1].to_s+" "+t[2].to_s+" "+t[3].to_s  )
    $export_file.puts( t[4].to_s+" "+t[5].to_s+" "+t[6].to_s+" "+t[7].to_s  )
    $export_file.puts( t[8].to_s+" "+t[9].to_s+" "+t[10].to_s+" "+t[11].to_s  )
    $export_file.puts( t[12].to_s+" "+t[13].to_s+" "+t[14].to_s+" "+t[15].to_s  )
    $export_file.puts( "" )
  
    end




#Is it a mirror transform?
def xformarray_is_mirror( t )

    mircount=0
    if t[0] <0 then mircount+=1 end
    if t[5] <0 then mircount+=1 end
    if t[10] <0 then mircount+=1 end

    if (mircount == 1 or mircount == 3) then return true end
    if (mircount == 0 or mircount == 2) then return false end

    end
























































# Updates the progrss bar
def UpdatePercentComplete()

    # Update progress bar
    $percentComplete=( 1.0-( $remainingEnts.to_f/$totalUsefulEntsInWorld.to_f) )*100

    $time_atthispoint=Time.now.to_f
    $time_elapsed=$time_atthispoint-$time_started

    #if  $time_elapsed.to_i == time_elapsed_previous.to_i
    #   return (0)
    #end

    if ( $percentComplete != 0)
       estimatemultiplier = 100/$percentComplete
    end

    estproctime = ( $time_elapsed.to_f * estimatemultiplier.to_f )

    elapsed=$time_elapsed
    est=estproctime-elapsed

    #if est.to_i==$previous_time_estimate and $percentComplete.to_i == $previous_percent_estimate
    if $percentComplete.to_i == $previous_percent_estimate
      $remainingEnts=$remainingEnts-1
      return (0)
    end

    if $blinker==1
       statstring=": "
       $blinker=0
    else
       statstring=". "
       $blinker=1
    end

    statstring+="Exporting to Hammer VMF format - "
    

    if $ParsingMode == "outputbrushes"
       statstring+="Writing Brushes:  "
    elsif $ParsingMode == "outputmodels"
       statstring+="Writing Models:   "
    elsif $ParsingMode == "outputentities"
       statstring+="Writing entities: "
    else
       statstring+="Writing         : "
       end

    statstring+=$percentComplete.to_i.to_s + "%"

    statstring+="      (ETA: "

    completestring=""
    if (est > 3599 )
          esthour = est/3600
          esthourint=esthour.floor
          completestring+= esthourint.to_s + " hours, "
          remain= est%3600
          estmin=(remain/60)
          completestring+= estmin.to_i.to_s + " minutes)"
    end
    if (est > 59 and est < 3600)
          estmin = est/60
          estmin=estmin.to_i
          completestring+= "about "+estmin.to_s + " minutes)"
    end
    if (est < 60 )
          esti=est.to_i
         completestring = esti.to_s + " seconds)"
    end
    statstring+=completestring

    Sketchup.set_status_text( statstring )

    $remainingEnts=$remainingEnts-1

    $previous_time_estimate = est.to_i
    $previous_percent_estimate = $percentComplete.to_i

    # To prevent overloads
    if ( $percentComplete > 101 )
       return (0)
    end
end





















# The code from the proof of concept, circa 2006
def theoldway( obj )

for ent in model.entities
    if( ent.is_a? Sketchup::Group )
          #Sketchup.set_status_text("Group Number: " + ent.entityID.to_s )
          entid=ent.entityID
          puts( "Group: \" " + entid.to_s + "\"" )

          $export_file.puts( "\tsolid" )
          $export_file.puts( "\t{" )
          $export_file.puts( "\t\t\"id\" \"" + ent.entityID.to_s + "\"" )

          trans = ent.transformation

          $groupmat = get_material_name(ent)
          #$export_file.puts( "YM: Material of Group: " + $groupmat )


          #Writes out the transform for debugging...
          #temptrans = (get_component_trans ent)
          #$export_file.puts("Translation x: " + temptrans[0].to_i.to_s + " y:" + temptrans[1].to_i.to_s + " z:" + temptrans[2].to_i.to_s )

          for ent_in_group in ent.entities
              if( ent_in_group.is_a? Sketchup::Face )
                   entid=ent_in_group.entityID

                   #puts( "Face: \" " + ent_in_group.entityID.to_s + "\"" )

                   myverts = ent_in_group.vertices

                   #This is a hack to get default material override concept to work
                   #TBA update to work as a function, more robustly, on multiple levels, with components, etc.
                   $mat = get_material_name(ent_in_group)
                   #$export_file.puts( "YM: Material of face: " + $mat )

                  if( $mat == "_unspecified_" )
                      $mat= $groupmat
                  end
                  if( $mat == "_unspecified_" )
                      $mat= defaultmaterialname
                  end

                  $export_file.puts( "\t\tside\n\t\t{\n")
                  $export_file.puts( "\t\t\t\"id\" \"" + ent_in_group.entityID.to_s + "\"\n" )

                  $tempstring = "\t\t\t\"plane\" \""
                  $myverts2=[]
                  $tempnormal=Geom::Vector3d.new
                  orig=Geom::Point3d.new

                  modelentities = model.active_entities

                  # Add empty groups to the entities in the model
                  tempgroup=modelentities.add_group
                  tempUVgroup=modelentities.add_group

                  # define the entities within the group
                  tempentities=tempgroup.entities
                  tempUVentities=tempUVgroup.entities

                  # Add a face to within the group
                  face = tempentities.add_face myverts #The whole face
                  #face = tempentities.add_face( myverts[0], myverts[1], myverts[2] ) #just a triangle

                  #uvorigin = tempUVentities.add_cpoint( orig ) #the UV origin
                  #uvorigin = tempUVentities.add_text("x", orig ) #the UV origin, cpoint not working...
                  uvorigin = tempUVentities.add_image("C:\\temp\\1REFTEX.jpg", orig, 0.2 ) #the UV origin, cpoint not working...

                  if debugAlert
                    UI.messagebox "Temp face made in local space"
                  end

                  #$export_file.puts( "Local origin: " + orig.to_s ) #should be 0,0,0

                  $templocalnormal = face.normal
#                  $export_file.puts( "Local Normal: " + $templocalnormal.to_s )

                  # Write out the verts of the new face
                  #i=0
                  #myverts.each do |pt|
                  #      $export_file.puts( "vert" + i.to_s + " local: " + pt.position.to_s )
                  #      i = i + 1
                  #end

                  ptz=Geom::Point3d.new( $templocalnormal.x, $templocalnormal.y, $templocalnormal.z )

                  visnorm = tempUVentities.add_line(ptz, orig)

                  if debugAlert
                    UI.messagebox "This is the local normal, or texture W axis"
                  end

                  uvtrans = Geom::Transformation.new( orig, $templocalnormal )
                  #$export_file.puts( "Local UV x axis: " + uvtrans.xaxis.to_s )
                  #$export_file.puts( "Local UV y axis: " + uvtrans.yaxis.to_s )

                  ptx=Geom::Point3d.new( uvtrans.xaxis.x, uvtrans.xaxis.y, uvtrans.xaxis.z )
                  pty=Geom::Point3d.new( uvtrans.yaxis.x, uvtrans.yaxis.y, uvtrans.yaxis.z )

                  visnormx = tempUVentities.add_line(ptx, orig)
                  #UI.messagebox "This is the U axis,"
                  visnormy = tempUVentities.add_line(pty, orig)
                  #UI.messagebox "This is the V axis,"

                  uvface = tempUVentities.add_face( ptx, pty, orig ) #just a triangle
                  uvverts = uvface.vertices
                  if debugAlert
                    UI.messagebox "The UV axes..."
                  end


                  # Write out the verts of the new face
                  i=0
                  uvverts.each do |pt|
                        #$export_file.puts( "UVs:\nVert" + i.to_s + " local: " + pt.position.to_s )
                        i = i + 1
                  end


                  tempgroup.transform!(trans)

                  if debugAlert
                    UI.messagebox "xform added to brush plane coordinates"
                  end

                  tempUVgroup.transform!(trans)
                  if debugAlert
                    UI.messagebox "xform added to UV coordinates"
                  end

                  tempents=tempgroup.explode
                  #UI.messagebox "xplode"

                  i=0
                  0.upto(tempents.length - 1) do |that|

                    if( tempents[i].is_a? Sketchup::Face )
                      $myverts2 = tempents[i].vertices

                      $tempnormal = tempents[i].normal
                      #$mat = get_material_name( tempents[i] )

                      #UI.messagebox $mat
                      #UI.messagebox $tempnormal
                      #UI.messagebox "found the new FACE"
                    end
                    i = i + 1
                  end

                  ent_arr=[]
                  tempents.each do |element|
                    if( element.typename == "Face" )
                        ent_arr.push element
                    end
                  end
                  tempents.each do |element|
                    if( element.typename == "Edge" )
                        ent_arr.push element
                    end
                  end

                 cnt = 2
                   0.upto(2) do |that|
                   #group.erase!
                        vert = Geom::Point3d.new( $myverts2[cnt].position.x, $myverts2[cnt].position.y, $myverts2[cnt].position.z )

                        tempstringx = sprintf( "%.5f",vert.x.to_f.to_inch )
                        tempstringy = sprintf( "%.5f",vert.y.to_f.to_inch )
                        tempstringz = sprintf( "%.5f",vert.z.to_f.to_inch )
                        $tempstring += "(" + tempstringx + " " + tempstringy + " " + tempstringz + ")"

                        if(cnt == 0)
                           $tempstring += "\""
                        else
                           $tempstring += " "
                        end
#                          $export_file.puts( tempstringx + " " + tempstringy + " " + tempstringz )
                        cnt = cnt - 1
                  end
                 #               UI.messagebox ent_arr

                    ent_arr.each do |e|
                         if not e.deleted?
                             e.erase!
                         end
                    end

                   $export_file.puts( $tempstring )


                  # $export_file.puts( "YM: Material of face- " + $mat )

                   $export_file.puts( "\t\t\t\"material\" \"" + $mat + "\"" )

                  #move the tempUVgroup back to the origin. Find the movement by extracting the image object origin

                  tempUVents=tempUVgroup.explode
                  if debugAlert
                    UI.messagebox "xplode UVs"
                  end


                  array_of_ents_to_delete=[]
                  uvverts=[]

                  tempUVents.each do |ent|
                    if( ent.typename == "Image" )
                        pt = ent.origin
                        #$export_file.puts( "The xformed UV origin: " + pt.to_s )
                        vec = Geom::Vector3d.new(pt.x, pt.y, pt.z)
                        $t = Geom::Transformation.translation(vec)
                        array_of_ents_to_delete.push(ent)
                    end
                    if( ent.typename == "Face" )
                        uvverts = ent.vertices
                        array_of_ents_to_delete.push(ent)
                    end
                    if( ent.typename == "Edge" )
                        array_of_ents_to_delete.push(ent)
                    end
                  end


	          newgroup = modelentities.add_group(array_of_ents_to_delete)

                 #UI.messagebox "Regroup it"

                  $t.invert!
                  newgroup.transform!($t)
                 #UI.messagebox "move it to the origin"

                 tempUVents.clear
                 uvverts.clear
                  tempUVents=newgroup.explode
                  #UI.messagebox "xplode UVs final"


                  scale_U=visnormx.length * global_texture_scale_factor
                  scale_V=visnormy.length * global_texture_scale_factor

                  array_of_ents_to_delete.clear

                  tempUVents.each do |ent|
                    if( ent.typename == "Image" )
                        array_of_ents_to_delete.push(ent)
                    end
                    if( ent.typename == "Face" )
                        uvverts = ent.vertices
                        array_of_ents_to_delete.push(ent)
                    end
                    if( ent.typename == "Edge" )
                        array_of_ents_to_delete.push(ent)

                        alertmessage0=sprintf( "%.3f",scale_U.to_f )
                        alertmessage1=sprintf( "%.3f",scale_V.to_f )
                        alertmessage2="Uscale: "+ alertmessage0 + "   Vscale: "+ alertmessage1

                        if debugAlert
                          UI.messagebox alertmessage2
                        end

                    end
                  end


                  # Write out the verts of the new face
                  i=0
                  uvverts.each do |pt|
                        #$export_file.puts( "vert" + i.to_s + " final fixed: " + pt.position.to_s )
                        i = i + 1
                  end


                   $tempstring = "\t\t\t\"uaxis\" \""

                        uxtemp=uvverts[0].position.x
                        tempstringx = sprintf( "%.5f",uxtemp.to_f.to_inch )
                        uytemp=uvverts[0].position.y
                        tempstringy = sprintf( "%.5f",uytemp.to_f.to_inch )
                        uztemp=uvverts[0].position.z
                        tempstringz = sprintf( "%.5f",uztemp.to_f.to_inch )

                        uScaletemp=scale_U
                        tempstringS = sprintf( "%.3f",uScaletemp.to_f )
                        $tempstring += "[" + tempstringx + " " + tempstringy + " " + tempstringz + " 0] "+ tempstringS +"\""


                   $export_file.puts( $tempstring )

                   $tempstring = "\t\t\t\"vaxis\" \""

                        uxtemp=uvverts[1].position.x * -1
                        tempstringx = sprintf( "%.3f",uxtemp )
                        uytemp=uvverts[1].position.y * -1
                        tempstringy = sprintf( "%.3f",uytemp )
                        uztemp=uvverts[1].position.z * -1
                        tempstringz = sprintf( "%.3f",uztemp )

                        vScaletemp=scale_V
                        tempstringS = sprintf( "%.3f",vScaletemp.to_f )
                        $tempstring += "[" + tempstringx + " " + tempstringy + " " + tempstringz + " 0] "+ tempstringS +"\""


                   $export_file.puts( $tempstring )

                   array_of_ents_to_delete.each do |e|
                         if not e.deleted?
                             e.erase!
                         end
                    end




# Non-working hack for UV's- delete

                  #tempUVents.each do |element|
                  #  if( element.typename == "Face" )
                  #      myUVverts = element.vertices
                  # Write out the verts of the new face
                  #i=0
                  #myUVverts.each do |pt|
                  #      $export_file.puts( "transformedUVs:\nVert" + i.to_s + " local: " + pt.position.to_s )
                  #      i = i + 1
                  #end
                  #      ent_arr.push element
                  #  end
                  #end

                  #mybignorm=Geom::Vector3d.new( $tempnormal )



                   mybignorm= $tempnormal

                   normtrans = Geom::Transformation.new( orig, $tempnormal )

                   uvtrans = normtrans

                   #$export_file.puts( normtrans.xaxis )
                   #$export_file.puts( normtrans.yaxis )
                   #$export_file.puts( normtrans.zaxis )


                   #UI.messagebox normtrans
                   #$export_file.puts( "normal: " + normal[0].to_s + " " + normal[1].to_s + " " + normal[2].to_s)

                   normx = normtrans.xaxis
                   normy = normtrans.yaxis
                   normz = normtrans.zaxis

                   #$export_file.puts( "normal x: " + normx.x.to_s + " " + normx.y.to_s + " " + normx.z.to_s )
                   #$export_file.puts( "normal y: " + normy.x.to_s + " " + normy.y.to_s + " " + normy.z.to_s )
                   #$export_file.puts( "normal z: " + normz.x.to_s + " " + normz.y.to_s + " " + normz.z.to_s )

                   $tempstring = "\t\t\t\"uaxis\" \""

                        uxtemp=uvtrans.xaxis.x
                        tempstringx = sprintf( "%.3f",uxtemp )
                        uytemp=uvtrans.xaxis.y
                        tempstringy = sprintf( "%.3f",uytemp )
                        uztemp=uvtrans.xaxis.z
                        tempstringz = sprintf( "%.3f",uztemp )
                        $tempstring += "[" + tempstringx + " " + tempstringy + " " + tempstringz + " 0] 0.25\""


                   #$export_file.puts( $tempstring )

                   $tempstring = "\t\t\t\"vaxis\" \""

                        vxtemp=uvtrans.yaxis.x
                        tempstringx = sprintf( "%.3f",vxtemp )
                        vxtemp=uvtrans.yaxis.y * -1
                        tempstringy = sprintf( "%.3f",vxtemp )
                        vxtemp=uvtrans.yaxis.z * -1
                        tempstringz = sprintf( "%.3f",vxtemp )
                        $tempstring += "[" + tempstringx + " " + tempstringy + " " + tempstringz + " 0] 0.25\""


#                   $export_file.puts( $tempstring )

                   if( normx == 0)
                       ang = 90
                   else
                       ang = 90
#                       ang = Math.atan2(normy, normx)
#                         ang = ( ( Math.atan2(normy, normx) )/ 0.017453293 )
#                         ang = (normy/normx)
                   end
#                     $export_file.puts( "Y over X: " + ang.to_s )



# [ texture shift ] Texture Scale factor

                   uam=(-1 * Math.sin(ang) )
                   ubm=Math.cos(ang)

 #                  $export_file.puts( "\t\t\t\"uaxis\" \"["+ normtrans.xaxis.x.to_s +" "+ normtrans.xaxis.y.to_s + " "+ normtrans.xaxis.z.to_s + " 0] 1\"" )

                   #rotation needs to be done...

                   $export_file.puts( "\t\t\t\"rotation\" \"0\"" )
                   $export_file.puts( "\t\t\t\"lightmapscale\" \"16\"" )
                   $export_file.puts( "\t\t\t\"smoothing_groups\" \"0\"" )

                   $export_file.puts( "\t\t}" )
               end
          end
#           end of solid
          $export_file.puts( "\t}" )

      end
  end
#      end of world
     $export_file.puts( "}" )

# PSB volumes are in world, but other entities, such as player starts, lights, come afterwards
# in the final version, just pick out objects as you go through them the first time, save them to an array, and print them out afterwards

for ent in model.entities
 if( ent.is_a? Sketchup::ComponentInstance )
   #$export_file.puts( "Component Identified" )
   defin=ent.definition
   compdefname = defin.name
   entid=ent.entityID
   #$export_file.puts( "Defenition is called: " + compdefname )

   if( compdefname == "info_player_start" )
      #The following should really be a function
      puts( "info_player_start: \" " + entid.to_s + "\"" )
      #$export_file.puts( "write out the " + compdefname + " code...")
      $export_file.puts( "entity" )
      $export_file.puts( "{" )
      $export_file.puts( "\t\"id\" \"" + ent.entityID.to_s + "\"" )
      $export_file.puts( "\t\"classname\" \"info_player_start\"" )
      $export_file.puts( "\t\"angles\" \"0 0 0\"" )

      trans = ent.transformation
      pt=trans.origin

      ox = sprintf( "%.3f",pt.x.to_i )
      oy = sprintf( "%.3f",pt.y.to_i )
      oz = sprintf( "%.3f",pt.z.to_i )

#      $export_file.puts( "x: " +pt.x.to_i.to_s )
#      $export_file.puts( "y: " +pt.y.to_i.to_s )
#      $export_file.puts( "z: " +pt.z.to_i.to_s )

      $export_file.puts( "\t\"origin\" \""+ox+" "+oy+" "+oz+ "\"" )

      $export_file.puts( "\teditor" )
      $export_file.puts( "\t{" )
      $export_file.puts( "\t\t\"color\" \"0 255 0\"" )
      $export_file.puts( "\t\t\"visgroupshown\" \"1\"" )
      $export_file.puts( "\t\t\"visgroupautoshown\" \"1\"" )
      $export_file.puts( "\t}" )
      $export_file.puts( "}" )


#      $export_file.puts( "origin: " + pt.to_s ) #exports the point to a string, but adds units


#      ox=trans.xaxis.to_s
#      oy=trans.yaxis.to_s
#      oz=trans.zaxis.to_s

#     trans = (get_component_trans ent)
#     $export_file.puts("translation x: " + trans[0].to_i.to_s + " y:" + trans[1].to_i.to_s + " z" + trans[2].to_i.to_s )


   end

   if( compdefname == "light" )
      #The following should really be a function
      puts( "Light: \" " + entid.to_s + "\"" )
      #$export_file.puts( "write out the " + compdefname + " code...")
      $export_file.puts( "entity" )
      $export_file.puts( "{" )
      $export_file.puts( "\t\"id\" \"" + ent.entityID.to_s + "\"" )
      $export_file.puts( "\t\"classname\" \"light\"" )
      $export_file.puts( "\t\"_light\" \"155 155 155\"" )
      $export_file.puts( "\t\"_lightHDR\" \"-1 -1 -1 1\"" )
      $export_file.puts( "\t\"_quadratic_attn\" \"1\"" )

      trans = ent.transformation
      pt=trans.origin

      ox = sprintf( "%.3f",pt.x.to_i )
      oy = sprintf( "%.3f",pt.y.to_i )
      oz = sprintf( "%.3f",pt.z.to_i )

      $export_file.puts( "\t\"origin\" \""+ox+" "+oy+" "+oz+ "\"" )

      $export_file.puts( "\teditor" )
      $export_file.puts( "\t{" )
      $export_file.puts( "\t\t\"color\" \"220 30 220\"" )
      $export_file.puts( "\t\t\"visgroupshown\" \"1\"" )
      $export_file.puts( "\t\t\"visgroupautoshown\" \"1\"" )
      $export_file.puts( "\t}" )
      $export_file.puts( "}" )
   end

   if( compdefname == "light_environment" )
      #The following should really be a function
      puts( "Light_Environment: \" " + entid.to_s + "\"" )
      #$export_file.puts( "write out the " + compdefname + " code...")
      $export_file.puts( "entity" )
      $export_file.puts( "{" )
      $export_file.puts( "\t\"id\" \"" + ent.entityID.to_s + "\"" )
      $export_file.puts( "\t\"classname\" \"light_environment\"" )
      $export_file.puts( "\t\"angles\" \"0 165 0\"" )
      $export_file.puts( "\t\"_light\" \"243 224 158 1200\"" )
      $export_file.puts( "\t\"_ambient\" \"194 236 252 175\"" )
      $export_file.puts( "\t\"_lightHDR\" \"-1 -1 -1 1\"" )
      $export_file.puts( "\t\"_ambientHDR\" \"-1 -1 -1 1\"" )
      $export_file.puts( "\t\"pitch\" \"-83\"" )

      trans = ent.transformation
      pt=trans.origin

      ox = sprintf( "%.3f",pt.x.to_i )
      oy = sprintf( "%.3f",pt.y.to_i )
      oz = sprintf( "%.3f",pt.z.to_i )

      $export_file.puts( "\t\"origin\" \""+ox+" "+oy+" "+oz+ "\"" )

      $export_file.puts( "\teditor" )
      $export_file.puts( "\t{" )
      $export_file.puts( "\t\t\"color\" \"220 30 220\"" )
      $export_file.puts( "\t\t\"visgroupshown\" \"1\"" )
      $export_file.puts( "\t\t\"visgroupautoshown\" \"1\"" )
      $export_file.puts( "\t}" )
      $export_file.puts( "}" )
   end

#      puts( "Unrecognized Component: # " + entid.to_s + " " + "def: "+ compdefname )
      # $export_file.puts( " -I don't recognize this component yet"+ compdefname)


 end #component instance
end #parsing all entities a second time

$export_file.close

#SketchUp Status Bar
Sketchup.set_status_text("Export to Hammer VMF format complete. ("+out_name+")" )

# Signal end of Undo sequence event
model.commit_operation
end

















#--------------------------------------------- Calculate global transformation

def get_component_trans (c)
return [ c.transformation.origin.x.to_inch, c.transformation.origin.y.to_inch, c.transformation.origin.z.to_inch ]
end


def calc_parent ( e )
	trans = e.transformation.to_a
	x = trans[12]
	y = trans[13]
	z = trans[14]

	return x,y,z
end




#--------------------------------------------- Delete temporary elements stored in the 0_HammerEnt layer (not used)

def delete_entity_cache

  model = Sketchup.active_model
  entity_array=[]

  model.entities.each do |element|

    if(element.layer.name == "0_HammerEnt")
      entity_array.push element
    end

  end

  entity_array.each {|e| e.erase! if not e.deleted?}

end




def output_header #-------------------------- Write out the file header

# "If any of this information is missing Hammer will simply recreate it from its current form." Consider only saving out
# what is known to be true, and let hammer fill the holes.

$export_file.puts( "//Hammertime .vmf Exporter 0.80" )
$export_file.puts( "//Copyright Valve Corporation, 2008" )
$export_file.puts( "//Exported from SketchUp "+$su_version.to_s )
$export_file.puts( "//Brush calculation mode set to: "+$brush_mode )
$export_file.puts( "//UV output set to: "+$uv_mode.to_s )
$export_file.puts( "\n" )

$export_file.puts( "versioninfo\n{" )
$export_file.puts( "\t\"editorversion\" \"400\"" )
$export_file.puts( "\t\"editorbuild\" \"3576\"" )
$export_file.puts( "\t\"mapversion\" \"1\"" )
$export_file.puts( "\t\"formatversion\" \"100\"" )
$export_file.puts( "\t\"prefab\" \"0\"" )
$export_file.puts( "}" )

# prefab (bool): Whether it is a full map or simply a collection of prefabricated objects.
# Used to define if the file contains an object like a sofa made from other brushes and entities, rather than a full blown map.


# Write Out Vis groups
$export_file.puts( "visgroups\n{\n}" )
#TBD: Save out layers as a monocline group of visgroups.

# Write Out View Settings
$export_file.puts( "viewsettings\n{\n" )
$export_file.puts( "\t\"bSnapToGrid\" \"1\"" )
$export_file.puts( "\t\"bShowGrid\" \"1\"" )
$export_file.puts( "\t\"bShowLogicalGrid\" \"0\"" )
$export_file.puts( "\t\"nGridSpacing\" \"64\"" )
$export_file.puts( "\t\"bShow3DGrid\" \"0\"" )
$export_file.puts( "}" )

if $debugOutput then $export_file.puts( "// *** [0] Begin World\n\n" ) end

# Write Out World
$export_file.puts( "world\n{" )
$export_file.puts( "\t\"id\" \"1\"" )
$export_file.puts( "\t\"mapversion\" \"1\"" )
$export_file.puts( "\t\"classname\" \"worldspawn\"" )
$export_file.puts( "\t\"skyname\" \"" +$defaultskyboxname + "\"" )

#not sure what these do, export works with and without them.
$export_file.puts( "\t\"maxpropscreenwidth\" \"-1\"" )
$export_file.puts( "\t\"detailvbsp\" \"detail.vbsp\"" )
$export_file.puts( "\t\"detailmaterial\" \"detail/detailsprites\"" )

end


# tried this so that old alg could close outliner. Unfortunately, it only works in windows,
#and doesn't help anyway, since the SU UI inspector API code doesn't reliably close it - Sometimes it opens it....

#require 'Win32API' #needed to call the user32.dll
#def isWindowVisible(name)
#    findWindow = Win32API.new("user32.dll", "FindWindow", ['P','P'],
#'N')
#    isWindowVisible= Win32API.new("user32.dll", "IsWindowVisible",
#['P'], 'N')
#    pw=findWindow.call(0,name)
#    return isWindowVisible.call(pw)==1
#end


#---------------------------------------------------------------------------------Menu/Loading

# Menu items
if( not file_loaded?("ht.rb") )
  add_separator_to_menu("Plugins")
  UI.menu("Plugins").add_item("Hammer Time!") { ht }
  UI.menu("Plugins").add_item("Hammer Time! (Slow)") { htslow }
end

#-----------------------------------------------------------------------------
file_loaded("ht.rb")
